import { useState, useRef, useEffect } from 'react'
import { useNavigate } from 'react-router-dom'
import { Apple, Target, TrendingUp, TrendingDown, Heart, AlertCircle, Calendar, CheckCircle, Award, Star, Users, Zap, Shield, Rocket, Sparkles } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedGoal, setSelectedGoal] = useState('')
  const [consultationDate, setConsultationDate] = useState('')
  const [showSuccess, setShowSuccess] = useState(false)

  const goals = [
    {
      id: 'weight-loss',
      name: 'Похудение',
      description: 'Снижение веса с сохранением мышечной массы',
      icon: TrendingDown,
      color: 'var(--wellness-cyan)'
    },
    {
      id: 'weight-gain',
      name: 'Набор массы',
      description: 'Увеличение веса за счет мышечной массы',
      icon: TrendingUp,
      color: 'var(--wellness-purple)'
    },
    {
      id: 'healthy-lifestyle',
      name: 'ЗОЖ',
      description: 'Поддержание здорового образа жизни',
      icon: Heart,
      color: 'var(--wellness-pink)'
    },
    {
      id: 'allergies',
      name: 'Аллергии',
      description: 'Питание при пищевых аллергиях и непереносимостях',
      icon: AlertCircle,
      color: 'var(--wellness-blue)'
    }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    
    if (!selectedGoal || !consultationDate) {
      alert('Пожалуйста, выберите цель и дату консультации')
      return
    }

    const goalData = goals.find(g => g.id === selectedGoal)
    
    const newConsultation = {
      id: Date.now(),
      plan: goalData.name,
      date: consultationDate,
      rating: null
    }

    const existing = JSON.parse(localStorage.getItem('nutriConsultations') || '[]')
    existing.push(newConsultation)
    localStorage.setItem('nutriConsultations', JSON.stringify(existing))

    setShowSuccess(true)
    setTimeout(() => {
      navigate('/')
    }, 2000)
  }

  return (
    <div className="enroll-page-wellness">
      <div className="container">
        <section className="enroll-header-wellness">
          <div className="header-badge-wellness">
            <Apple size={18} />
            <span>ЗАПИСЬ НА КОНСУЛЬТАЦИЮ</span>
          </div>
          <h1 className="enroll-title-wellness">
            Записаться на <span className="title-accent-wellness">консультацию</span>
          </h1>
          <p className="enroll-description-wellness">
            Выберите вашу цель и дату консультации с профессиональным нутрициологом
          </p>
        </section>

        <section className="enroll-form-section-wellness">
          <form className="form-wellness" onSubmit={handleSubmit}>
            <div className="goals-header-wellness">
              <h2 className="goals-title-wellness">Выберите вашу цель</h2>
              <p className="goals-description-wellness">
                Каждая цель требует индивидуального подхода к питанию
              </p>
            </div>
            <div className="goals-selection-wellness">
              {goals.map((goal) => {
                const Icon = goal.icon
                return (
                  <button
                    key={goal.id}
                    type="button"
                    className={`goal-card-wellness ${selectedGoal === goal.id ? 'selected' : ''}`}
                    onClick={() => setSelectedGoal(goal.id)}
                    style={{ '--goal-color': goal.color }}
                  >
                    <div className="goal-icon-wrapper-wellness">
                      <Icon size={36} />
                    </div>
                    <h3 className="goal-name-wellness">{goal.name}</h3>
                    <p className="goal-desc-wellness">{goal.description}</p>
                    {selectedGoal === goal.id && (
                      <div className="check-badge-wellness">
                        <CheckCircle size={20} />
                      </div>
                    )}
                  </button>
                )
              })}
            </div>

            {selectedGoal && (
              <div className="date-section-wellness">
                <div className="date-label-wellness">
                  <Calendar size={18} />
                  <span>Дата консультации</span>
                </div>
                <input
                  type="date"
                  className="date-input-wellness"
                  value={consultationDate}
                  onChange={(e) => setConsultationDate(e.target.value)}
                  min={new Date().toISOString().split('T')[0]}
                  required
                />
              </div>
            )}

            <button 
              type="submit" 
              className="submit-btn-wellness"
              disabled={!selectedGoal || !consultationDate}
            >
              <CheckCircle size={18} />
              <span>Записаться на консультацию</span>
            </button>

            {showSuccess && (
              <div className="success-message-wellness">
                <Award size={20} />
                <span>Запись успешно оформлена!</span>
              </div>
            )}
          </form>
        </section>

        <ImageSlider />

        <ParallaxSection />

        <ProcessSection />
      </div>
    </div>
  )
}

const ParallaxSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)
  const [mousePosition, setMousePosition] = useState({ x: 0, y: 0 })

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    const handleMouseMove = (e) => {
      if (sectionRef.current) {
        const rect = sectionRef.current.getBoundingClientRect()
        setMousePosition({
          x: (e.clientX - rect.left) / rect.width,
          y: (e.clientY - rect.top) / rect.height
        })
      }
    }

    if (sectionRef.current) {
      sectionRef.current.addEventListener('mousemove', handleMouseMove)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
        sectionRef.current.removeEventListener('mousemove', handleMouseMove)
      }
    }
  }, [])

  const testimonials = [
    {
      name: 'Анна К.',
      text: 'За 3 месяца сбросила 8 кг! План питания был идеальным.',
      rating: 5,
      gradient: 'linear-gradient(135deg, #6C5CE7 0%, #A29BFE 100%)'
    },
    {
      name: 'Михаил С.',
      text: 'Набрал мышечную массу благодаря правильному питанию.',
      rating: 5,
      gradient: 'linear-gradient(135deg, #00B894 0%, #55EFC4 100%)'
    },
    {
      name: 'Елена П.',
      text: 'Решила проблемы с пищеварением. Очень благодарна!',
      rating: 5,
      gradient: 'linear-gradient(135deg, #FD79A8 0%, #FDCB6E 100%)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`parallax-section-wellness ${isVisible ? 'visible' : ''}`}
      style={{
        '--mouse-x': mousePosition.x,
        '--mouse-y': mousePosition.y
      }}
    >
      <div className="container">
        <div className="parallax-header-wellness">
          <h2 className="parallax-title-wellness">Истории успеха</h2>
          <p className="parallax-description-wellness">
            Реальные результаты наших клиентов
          </p>
        </div>
        <div className="parallax-grid-wellness">
          {testimonials.map((testimonial, index) => (
            <div 
              key={index} 
              className="parallax-card-wellness"
              style={{ 
                animationDelay: `${index * 0.15}s`,
                background: testimonial.gradient,
                transform: `translate(
                  calc(var(--mouse-x) * ${(index - 1) * 20}px),
                  calc(var(--mouse-y) * ${(index - 1) * 20}px)
                ) rotateX(calc(var(--mouse-y) * 5deg)) rotateY(calc(var(--mouse-x) * 5deg))`
              }}
            >
              <div className="parallax-content-wellness">
                <div className="parallax-stars-wellness">
                  {Array.from({ length: 5 }).map((_, i) => (
                    <Star
                      key={i}
                      size={16}
                      fill="#FDCB6E"
                      color="#FDCB6E"
                    />
                  ))}
                </div>
                <p className="parallax-text-wellness">"{testimonial.text}"</p>
                <div className="parallax-author-wellness">{testimonial.name}</div>
              </div>
            </div>
          ))}
        </div>
      </div>
    </section>
  )
}

const ProcessSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const steps = [
    {
      icon: Rocket,
      title: 'Консультация',
      description: 'Первичная встреча с нутрициологом для анализа ваших потребностей',
      step: '01',
      color: 'var(--wellness-purple)'
    },
    {
      icon: Target,
      title: 'План питания',
      description: 'Разработка индивидуального плана питания под ваши цели',
      step: '02',
      color: 'var(--wellness-cyan)'
    },
    {
      icon: Sparkles,
      title: 'Результаты',
      description: 'Отслеживание прогресса и корректировка плана при необходимости',
      step: '03',
      color: 'var(--wellness-pink)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`process-section-wellness ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="process-header-wellness">
          <h2 className="process-title-wellness">Как это работает</h2>
          <p className="process-description-wellness">
            Простой процесс от консультации до результата
          </p>
        </div>
        <div className="process-grid-wellness">
          {steps.map((step, index) => {
            const Icon = step.icon
            return (
              <div 
                key={index} 
                className="process-step-wellness"
                style={{ 
                  animationDelay: `${index * 0.15}s`,
                  '--step-color': step.color
                }}
              >
                <div className="process-step-number-wellness">{step.step}</div>
                <div className="process-step-icon-wellness">
                  <Icon size={32} />
                </div>
                <h3 className="process-step-title-wellness">{step.title}</h3>
                <p className="process-step-text-wellness">{step.description}</p>
              </div>
            )
          })}
        </div>
      </div>
    </section>
  )
}

export default Enroll
